from typing import Any, Optional

from aiohttp import web
from aiopg.sa.engine import Engine

from sendr_aiohttp import BaseUrlDispatcher
from sendr_qlog.http.aiohttp import signal_request_id_header

from {{cookiecutter.import_path}} import __version__
from {{cookiecutter.import_path}}.api.middlewares import (
    middleware_exceptions_handler, middleware_logging_adapter, middleware_stats
)
from {{cookiecutter.import_path}}.api.routes.user import USER_ROUTES
from {{cookiecutter.import_path}}.api.routes.utility import UTILITY_ROUTES
from {{cookiecutter.import_path}}.conf import settings
from {{cookiecutter.import_path}}.interactions.base import BaseInteractionClient, create_connector
from {{cookiecutter.import_path}}.utils.db import create_configured_engine


class {{cookiecutter.ProjectName}}Application(web.Application):
    _urls = (
        UTILITY_ROUTES,
        USER_ROUTES,
    )

    def __init__(self, db_engine: Optional[Engine] = None):
        super().__init__(
            router=BaseUrlDispatcher(),
            middlewares=(
                middleware_stats,
                middleware_logging_adapter,
                middleware_exceptions_handler,
            ),
        )

        if db_engine:
            self.db_engine = db_engine
        else:
            self.on_startup.append(self.open_engine)
            self.on_cleanup.append(self.close_engine)

        self.add_routes()
        self.add_sentry()
        self.on_response_prepare.append(signal_request_id_header)
        self.on_startup.append(self.create_connector)
        self.on_cleanup.append(self.close_engine)
        self.on_cleanup.append(self.close_connector)

    def add_routes(self) -> None:
        for routes in self._urls:
            self.router.add_routes(routes)

    def add_sentry(self) -> None:
        from sendr_qlog.sentry import sentry_init
        if settings.SENTRY_DSN:
            self.on_cleanup.append(sentry_init(settings.SENTRY_DSN, release=__version__))

    async def open_engine(self, _: Any) -> None:
        self.db_engine = create_configured_engine()

    async def close_engine(self, _: web.Application) -> None:
        if self.db_engine:
            self.db_engine.close()
        await self.db_engine.wait_closed()

    async def create_connector(self, _: Any) -> None:
        BaseInteractionClient.CONNECTOR = create_connector()

    async def close_connector(self, _: Any) -> None:
        await BaseInteractionClient.close_connector()
